/** @file   scatterevent.h
 * @brief   Declaration of ScatterEvent - classes.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_SCATTEREVENT_H
#define H_WWW_SCATTEREVENT_H

#include "rect2d.h"
#include "dynamicevent.h"

namespace WeWantWar {


/** @class  ScatterEvent
 * @brief   Removes GameObjects during the gameplay.
 * @author  Tomi Lamminsaari
 *
 * With ScatterEvent you can delete GameObjects and other elements from
 * level. There can be several rectangular areas where we scatter the
 * elements.
 *
 * The scattering information will be read from Level Description File.
 */
class ScatterEvent : public DynamicEvent
{
public:

  ///
  /// Static members and structures
  /// =============================
  
  /** Datatype we use when defining the scattering mode. */
  typedef int ScatterMode;
  
  /** Removes all the objects that are instances of GameObject */
  static const ScatterMode SCATTER_GAMEOBJECTS = 0;
  /** Removes only killed objects. */
  static const ScatterMode SCATTER_KILLED = 1;
  /** Removes only objects with certain id-code. */
  static const ScatterMode SCATTER_ID = 2;
  /** Removes the doors. */
  static const ScatterMode SCATTER_DOOR = 3;
  /** Removes the instances of BonusObject */
  static const ScatterMode SCATTER_BONUS = 4;
  
  /** Each Scatter-event can have several independent items that it
   * removes. Each of them is located in structures of this type.
   */
  struct ScatterItem {
    /** The mode we use for this element. */
    ScatterMode mode;
    /** The rectangle that defines the area where the scattering will be
     * located.
     */
    eng2d::Rect2D rect;
    /** An id-code of the objects being removed when using SCATTER_ID - mode. */
    int idcode;
  };
  
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================
  
  /** Constructor.
   */
  ScatterEvent();
  
  /** Destructor.
   */
  virtual ~ScatterEvent();
  
  
  ///
  /// Methods
  /// =======
  
  /** Reimplemented readData-method. Reads the information of the scatter-event
   * This method expects that the "scatter" -tag has already been read away
   * from the stream.
   * @param     fin               The input stream
   * @return    0 if successful or nonzero if fails.
   */
  virtual int readData( std::istream& fin );
  
  /** Activates this event. This means that the possible timer starts to count
   * down. When the timer reaches the negative value, we delete the GameObjects.
   *
   * NOTE! This method does not scatter the GameObjects. This just enables
   * the countdown. You must keep calling the 'update()'-method.
   */
  virtual void activate();
  
  
  /** Reimplemented update-method. If this event has not been actived yet, this
   * method does nothing. Once this event is activated, this method does the
   * countdown and as the counter value reaches a negative number, this method
   * deletes the GameObjects.
   */
  virtual void update();
  
  
protected:

  ///
  /// Protected methods
  /// =================
  
  /** Does the scattering. This gets normally called by the 'update()'-method.
   */
  void scatter();
  
  /** Deletes the gameobjects. Every GameObject that is inside the rectanle
   * will be deleted.
   * @param     pSI               Pointer to ScatterItem that holds the
   *                              information.
   */
  void scatterGameObjects( ScatterItem* pSI );
  
  /** Deletes the killed gamobjects. Only killed objects inside the rectangle
   * will be deleted.
   * @param     pSI               Pointer to ScatterItem that holds the
   *                              information.
   */
  void scatterKilled( ScatterItem* pSI );
  
  /** Deletes the objects with certain id-code.
   * @param     pSI               Pointer to ScatterItem that holds the
   *                              information.
   */
  void scatterID( ScatterItem* pSI );
  
  /** Deletes the doors that are inside the rectangle.
   * @param     pSI               Pointer to ScatterItem that holds the
   *                              information.
   */
  void scatterDoors( ScatterItem* pSI );
  
  /** Deletes the bonusobjects inside the rectangle.
   * @param     pSI               Pointer to ScatterItem that holds the
   *                              information.
   */
  void scatterBonus( ScatterItem* pSI );
  
  /** Reads the data of SCATTER_GAMEOBJECTS - mode event.
   * @param     rIn               The input stream
   * @return    Pointer to ScatterItem or nullpointer if failed.
   */
  ScatterItem* readScatterGameObjects( std::istream& rIn );
  
  /** Reads the data of SCATTER_KILLED - mode event
   * @param     rIn               The input stream
   * @return    Pointer to ScatterItem or nullpointer if failed.
   */
  ScatterItem* readScatterKilled( std::istream& rIn );
  
  /** Reads the data of SCATTER_ID - mode event.
   * @param     rIn               The input stream
   * @return    Pointer to ScatterItem or nullpointer if failed.
   */
  ScatterItem* readScatterID( std::istream& rIn );
  
  /** Reads the data of SCATTER_DOOR - mode event.
   * @param     rIn               The input stream
   * @return    Pointer to ScatterItem or nullpointer if failed.
   */
  ScatterItem* readScatterDoor( std::istream& rIn );
  
  /** Reads the data of SCATTER_BONUS - mode event.
   * @param     rIn               The input stream
   * @return    Pointer to ScatterItem or nullpointer if failed.
   */
  ScatterItem* readScatterBonus( std::istream& rIn );
  
  
  ///
  /// Members
  /// =======
  
  /** List of the rectangles that together form the scattering area. */
  std::vector<ScatterItem*> m_scatterList;
};

};  // end of namespace

#endif // SCATTEREVENT_H

/**
 * Version history
 * ===============
 * $Log: scatterevent.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:13+02  lamminsa
 * Initial revision
 *
 */
